<?php

namespace App\Http\Controllers;

use App\Models\TaxesModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TaxController extends Controller
{
    public function index(Request $request)
    {
        $taxes = TaxesModel::where('branch_id', session('branch_id'))->orderBy('id', 'desc')
            ->get();

        return view('taxes.index', compact('taxes'));
    }

    public function add()
    {
        return view('taxes.add');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'taxName' => 'required|string|max:255',
            'taxValue' => 'required|numeric|max:150',
            'isDefault' => 'required|in:1,0',
            'show_flg' => 'sometimes|in:Y,N'
        ]);

        DB::beginTransaction();

        try {

            TaxesModel::create([
                'name' => $validated['taxName'],
                'value' => $validated['taxValue'],
                'is_default' => $validated['isDefault'],
                'show_flg' => $validated['show_flg'] ?? 'N',
                'branch_id' => session('branch_id')
            ]);

            DB::commit();

            return redirect()->route('tax-index')
                ->with('success', 'Tax added successfully!');
        } catch (\Exception $e) {

            DB::rollBack();
            \Log::error('Error while storing tax: ' . $e->getMessage());

            return redirect()->back()
                ->with('error', 'Something went wrong while adding the tax. Please try again.')
                ->withInput();
        }
    }


    public function edit($id)
    {
        $tax = TaxesModel::findOrFail($id);

        return view('taxes.edit', compact('tax'));
    }

    public function update(Request $request)
    {
        $validated = $request->validate([
            'tax_id' => 'required|exists:taxes,id',
            'taxName' => 'required|string|max:255',
            'taxValue' => 'required|numeric|max:150',
            'isDefault' => 'required|in:1,0',
            'show_flg' => 'sometimes|in:Y,N'
        ]);
        DB::beginTransaction();

        try {

            $tax = TaxesModel::findOrFail($validated['tax_id']);

            $tax->update([
                'name' => $validated['taxName'],
                'value' => $validated['taxValue'],
                'is_default' => $validated['isDefault'],
                'show_flg' => $validated['show_flg'] ?? 'N'
            ]);

            DB::commit();

            return redirect()->route('tax-index')
                ->with('success', 'Tax updated successfully!');

        } catch (\Exception $e) {

            DB::rollBack();
            \Log::error('Tax update failed: ' . $e->getMessage());

            return redirect()->back()
                ->with('error', 'Failed to update tax. Please try again.')
                ->withInput();
        }
    }
    public function delete($id)
    {
        $tax = TaxesModel::findOrFail($id);
        $tax->delete();

        return redirect()->route('tax-index')
            ->with('success', 'Tax deleted successfully!');
    }
}