<?php

namespace App\Http\Controllers;

use App\Models\BranchModel;
use App\Models\ClientModel;
use App\Models\InvoiceItemModel;
use App\Models\InvoiceModel;
use App\Models\ProductModel;
use App\Models\QuotesItemsModel;
use App\Models\QuotesModel;
use App\Models\User;
use App\Models\TaxesModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Carbon\Carbon;
use Symfony\Contracts\Service\Attribute\Required;

class QuotesController extends Controller
{
    public function index(Request $request)
    {
        $quotes = QuotesModel::with('client.user')->where('branch_id',session('branch_id'))->orderBy('id', 'desc')->get();

        return view('quotes.index', compact('quotes'));
    }
    public function add()
    {
        $branch_id = session('branch_id');

        $clients = ClientModel::with('user')->get();
        $products = ProductModel::all();

        $discount_types = DB::table('discount_type')->get();
        $taxes = TaxesModel::where('is_default', 1)->where('branch_id', $branch_id)->get();

        return view('quotes.add', compact('clients', 'products', 'discount_types', 'taxes'));
    }

    public function store(Request $request)
    {

        $branch_id = session('branch_id');
        $request->validate([
            'dynamicFields.*.qty' => 'required|numeric|lte:dynamicFields.*.available_qty',
            'dynamicFields.*.unit_price' => 'required|numeric',
            'dynamicFields.*.amount' => 'required|numeric',
            'client_num' => 'required',
            'client_id' => [
                Rule::requiredIf(function () use ($request) {
                    return !User::where('phone', $request->client_num)->exists();
                }),
            ],
        ]);

        DB::beginTransaction();
        try {
            $user = User::where('phone', $request->client_num)->first();

            $client_id = $this->saveClient($request, $user);

            if (!$client_id) {
                return redirect()->back()->with('error', 'Failed to create client');
            }

            $lastQuotes = QuotesModel::orderBy('id', 'desc')->first();
            $quote_id = $this->generateQuoteId($lastQuotes);

            $quote = QuotesModel::create([
                'quote_id' => $quote_id,
                'client_id' => $client_id,
                'quote_date' => $request->quote_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type ?? null,
                'discount' => $request->discount ?? null,
                'note' => $request->note,
                'term' => $request->term,
                'branch_id' => $branch_id,
                'status' => $request->status ?? '',
            ]);

            foreach ($request->products as $field) {
                $product = ProductModel::find($field['id']);

                QuotesItemsModel::create([
                    'quote_id' => $quote->id,
                    'product_id' => $field['id'],
                    'product_name' => $product->name ?? '',
                    'available_quantity' => $product->qty,
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'total' => $field['amount'],
                ]);
            }

            DB::commit();

            return redirect()->route('quote-pdf', ['id' => $quote->id])->with('success', 'Quote created successfully');

        } catch (\Exception $e) {

            DB::rollBack();

            \Log::info('While Add  Quotes' . $e->getMessage());

            return redirect()->back()->with('error', 'Failed to add quote: ' . $e->getMessage())->withInput();
        }
    }


    public function edit($id, $type)
    {
        $branch_id = session('branch_id');

        $clients = ClientModel::with('user')->get();

        $products = ProductModel::all();

        $discount_types = DB::table('discount_type')->get();

        $quote = QuotesModel::with('quotesItem', 'client.user')->findOrFail($id);

        $products = ProductModel::all();

        return view('quotes.edit', compact('clients', 'products', 'discount_types', 'quote', 'type'));
    }

    public function update(Request $request)
    {
        if ($request->type == 'convert') {
            $this->processConvert($request);
        } else {


            $request->validate([
                'dynamicFields.*.qty' => 'required|numeric|lte:dynamicFields.*.available_qty',
                'dynamicFields.*.unit_price' => 'required|numeric',
                'dynamicFields.*.amount' => 'required|numeric',
            ]);
            try {
                $quote = QuotesModel::findOrFail($request->id);

                $quote->update([
                    'client_id' => $request->client_id,
                    'quote_date' => $request->quote_date,
                    'due_date' => $request->due_date,
                    'amount' => $request->sub_total,
                    'final_amount' => $request->total_amount,
                    'discount_type' => $request->discount_type ?? null,
                    'discount' => $request->discount ?? null,
                ]);

                $productIds = [];

                foreach ($request->products as $field) {
                    $product = ProductModel::find($field['id']);

                    $quoteItem = QuotesItemsModel::updateOrCreate(
                        [
                            'quote_id' => $quote->id,
                            'product_id' => $field['id'],
                        ],
                        [
                            'product_name' => $product->name ?? '',
                            'available_quantity' => $product->qty,
                            'quantity' => $field['qty'],
                            'price' => $field['unit_price'],
                            'total' => $field['amount'],
                        ]
                    );

                    $productIds[] = $field['id'];
                }

                // Remove deleted items
                QuotesItemsModel::where('quote_id', $quote->id)
                    ->whereNotIn('product_id', $productIds)
                    ->delete();

                return redirect()->route('quotes-index')
                    ->with('success', 'Quote updated successfully');
            } catch (\Exception $e) {

                DB::rollBack();

                \Log::info('While Update  Quotes' . $e->getMessage());

                return redirect()->back()->with('error', 'Failed to update quote: ' . $e->getMessage())->withInput();
            }
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            $quote = QuotesModel::with('quotesItem.quoteItemTaxes')->findOrFail($id);

            $quote->quotesItem->each(function ($item) {
                $item->quoteItemTaxes()->delete();
            });

            $quote->quotesItem()->delete();
            $quote->delete();

            DB::commit();

            return redirect()->route('quotes-index')->with('success', 'Quote deleted successfully');

        } catch (\Exception $e) {

            DB::rollBack();

            \Log::info('While Deleting Quote' . $e->getMessage());

            return redirect()->route('quotes-index')->with('error', 'An error occurred while deleting the quote');
        }
    }

    public function processConvert($request)
    {

        $request->validate([
            'dynamicFields.*.qty' => 'required|numeric|lte:dynamicFields.*.available_qty',
            'dynamicFields.*.unit_price' => 'required|numeric',
            'dynamicFields.*.amount' => 'required|numeric',
        ]);

        try {
            DB::beginTransaction();

            $quote = QuotesModel::findOrFail($request->id);
            $lastInvoice = InvoiceModel::orderBy('id', 'desc')->first();
            $invoice_id = $this->generateInvoiceId($lastInvoice, $quote->client_id);

            $invoice = InvoiceModel::create([
                'invoice_id' => $invoice_id,
                'client_id' => $quote->client_id,
                'invoice_date' => $request->quote_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type,
                'discount' => $request->discount,
                'note' => $request->note,
                'term' => $request->term,
                'template_id' => $request->template_id,
                'branch_id' => session('branch_id'),
                'gst' => 'N',
                'quote_id' => $quote->quote_id,
                'status' => 1,
                'created_by' => Auth::user()->id,
            ]);

            foreach ($request->products as $field) {

                $product = ProductModel::find($field['id']);
                $available_qty = $product->qty - $field['qty'];

                if ($product) {
                    $product->update(['qty' => $available_qty]);
                }

                InvoiceItemModel::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $field['id'],
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'total' => $field['amount'],
                    'product_name' => $product->name ?? '',
                ]);
            }

            $quote->update(['oprntl_flag' => 'C']);

            DB::commit();

            return redirect()->route('quotes-index')->with('success', 'Quote converted to invoice successfully');

        } catch (\Exception $e) {

            DB::rollBack();

            \Log::info('While Convert to Quotes to Invoice' . $e->getMessage());

            return redirect()->back()->with('error', 'Failed to convert quote')->withInput();
        }
    }

    protected function generateQuoteId($lastQuotes)
    {
        if ($lastQuotes) {
            $lastQuotesId = $lastQuotes->quote_id;
            $parts = explode('-', $lastQuotesId);
            $prefix = $parts[0];
            $number = (int) $parts[1];
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }
        return 'OSQOT-001';
    }

    protected function generateInvoiceId($lastInvoice, $client_id)
    {
        if ($lastInvoice) {
            $lastInvoiceId = $lastInvoice->invoice_id;
            $parts = explode('-', $lastInvoiceId);
            $prefix = $parts[0];
            $number = (int) $parts[1];
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }

        $client = ClientModel::find($client_id);
        if ($client) {
            $company = BranchModel::where('id', $client->branch_id)->first();
            return $company ? $company->prefix . '-001' : 'INV-001';
        }
        return 'INV-001';
    }

    protected function saveClient(Request $request, $user = null)
    {
        try {
            if ($user) {
                $client = ClientModel::where('user_id', $user->id)->first();
                return $client->id;
            }

            $randomEmail = Str::random(10) . '@billing.com';
            $user = User::create([
                'name' => $request->client_id,
                'email' => $randomEmail,
                'phone' => $request->client_num,
                'password' => Hash::make('client@123'),
                'user_flg' => 'C',
                'active_status' => 'A',
                'branch_id' => session('branch_id')
            ]);

            $client = ClientModel::create([
                'user_id' => $user->id,
                'branch_id' => session('branch_id'),
                'active_status' => 'A',
            ]);

            return $client->id;
        } catch (\Exception $e) {
            \Log::info('While create Client:' . $e);
            return null;
        }
    }
}