<?php

namespace App\Http\Controllers;

use App\Models\PaymentStatusModel;
use App\Models\ProductDetailsModel;
use App\Models\ProductPaymentModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProductPaymentController extends Controller
{

    public function index(Request $request)
    {
        $payments_data = ProductDetailsModel::with('payments')->where('branch_id', session('branch_id'))
            ->has('payments')->orderBy('product_details_sid', 'desc')
            ->get();
        $payment_status = PaymentStatusModel::all();

        return view('product-payments.index', compact('payments_data', 'payment_status'));
    }

    public function add()
    {
        $payments = ProductPaymentModel::leftJoin('product_details', 'product_details.product_details_sid', '=', 'product_payments.product_dtl_id')
            ->selectRaw('SUM(product_payments.paid_amount) as total_amount')
            ->groupBy('product_payments.product_dtl_id', 'product_details.product_details_sid', 'product_payments.payment_date', 'product_payments.payment_type')
            ->get();

        $product_dtls = ProductDetailsModel::doesntHave('payments')->get();

        $all_product_dtls = ProductDetailsModel::all();

        return view('product-payments.add', compact('payments', 'product_dtls', 'all_product_dtls'));
    }
    public function getProductDetails($id)
    {
        $product_dtl = ProductDetailsModel::where('product_details_sid', $id)->first();

        if ($product_dtl) {
            return response()->json([
                'dueAmount' => round($product_dtl->total_amount, 2),
                'totalAmount' => round($product_dtl->total_amount, 2)
            ]);
        }

        return response()->json(['error' => 'Product not found'], 404);
    }

    public function store(Request $request)
    {
        $request->validate([
            'product_dtl_id' => 'required',
            'payment_type' => 'required',
            'payment_date' => 'required|date',
            'total_amount' => 'required|numeric',
            'amount' => 'required|numeric',
        ]);

        $product_payment = ProductPaymentModel::create([
            'product_dtl_id' => $request->product_dtl_id,
            'payment_type' => $request->payment_type,
            'payment_date' => $request->payment_date,
            'total_amount' => $request->total_amount,
            'paid_amount' => $request->amount,
        ]);

        $product_dtls = ProductDetailsModel::find($product_payment->product_dtl_id);
        $dueAmount = $product_dtls->total_amount;

        if ($request->amount >= $dueAmount) {
            $status = 'Paid';
        } elseif ($request->amount > 0 && $request->amount < $dueAmount) {
            $status = 'Partially Paid';
        } else {
            $status = 'Unpaid';
        }

        $paymentStatus = DB::table('payment_status')->where('name', $status)->first();
        $product_dtls->where('product_details_sid', $product_payment->product_dtl_id)
            ->update([
                'status' => $paymentStatus->id,
                'updated_at' => now(),
            ]);

        return redirect()->route('product-payment-index')
            ->with('success', 'Payment created successfully!');
    }

    public function edit($id)
    {
        $product = ProductDetailsModel::find($id);
        $payment = ProductPaymentModel::where('product_dtl_id', $id)->first();

        if (!$payment) {
            return redirect()->route('product-payments.index')
                ->with('error', 'Payment not found');
        }

        $totalAmount = $product->total_amount;
        $totalPaidAmount = ProductPaymentModel::where('product_dtl_id', $id)->sum('paid_amount');
        $dueAmount = round($totalAmount - $totalPaidAmount, 2);
        $paymentsForProduct = ProductPaymentModel::with('product_dtls')->where('product_dtl_id', $id)->get();

        return view('product-payments.edit', compact(
            'payment',
            'product',
            'totalAmount',
            'totalPaidAmount',
            'dueAmount',
            'paymentsForProduct'
        ));
    }

    public function update(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric',
            'payment_type' => 'required',
            'payment_date' => 'required|date',
        ]);

        $id = $request->payment_id;

        if ((float) $request->totalAmount < (float) $request->amount + (float) $request->paidAmount) {
            return back()->with('error', 'Amount should be lesser than Total amount, Please try again!');
        }

        ProductPaymentModel::create([
            'product_dtl_id' => $id,
            'payment_type' => $request->payment_type,
            'total_amount' => $request->total_amount,
            'paid_amount' => $request->amount,
            'payment_date' => $request->payment_date,
        ]);

        $paidAmount = ProductPaymentModel::where('product_dtl_id', $id)->sum('paid_amount');
        $product_dtl = ProductDetailsModel::where('product_details_sid', $id)->first();

        if ($product_dtl) {
            if ($paidAmount >= $product_dtl->total_amount) {
                $status = 'Paid';
            } elseif ($paidAmount > 0 && $paidAmount < $product_dtl->total_amount) {
                $status = 'Partially Paid';
            } else {
                $status = 'Unpaid';
            }

            $paymentStatus = DB::table('payment_status')->where('name', $status)->first();
            if ($paymentStatus) {
                $product_dtl->update([
                    'status' => $paymentStatus->id,
                    'updated_at' => now(),
                ]);
            }
        }

        return redirect()->route('product-payment-index')
            ->with('success', 'Payment updated and a new record created successfully!');
    }

    public function delete($id)
    {
        $payments = ProductPaymentModel::where('product_dtl_id', $id)->get();

        if ($payments->isNotEmpty()) {
            $product_dtls = ProductDetailsModel::find($id);
            if ($product_dtls) {
                $product_dtls->status = 1;
                $product_dtls->save();
            }

            ProductPaymentModel::where('product_dtl_id', $id)->delete();

            return redirect()->route('product-payment-index')
                ->with('success', 'Payments deleted successfully!');
        }

        return redirect()->route('product-payment-index')
            ->with('error', 'Payments not found!');
    }
}