<?php

namespace App\Http\Controllers;

use App\Models\InvoiceItemModel;
use App\Models\ProductDetailsModel;
use App\Models\ProductHistoryModel;
use App\Models\ProductPaymentModel;
use App\Models\ProductModel;
use App\Models\CategoryModel;
use App\Models\QuotesItemsModel;
use App\Models\SupplierModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;


class ProductController extends Controller
{
    protected $branch_id;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->branch_id = session('branch_id');
            return $next($request);
        });
    }

    public function index(Request $request)
    {
        $products = ProductModel::with('user')->where('branch_id', $this->branch_id)->orderBy('id', 'desc')
            ->get();

        return view('products.index', compact('products'));
    }
    public function add(Request $request)
    {
        $categories = CategoryModel::where('branch_id', $this->branch_id)->get();
        $suppliers = SupplierModel::where('branch_id', $this->branch_id)->where('active_status', 1)->get();

        return view('products.add', compact('categories', 'suppliers'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'productCode' => 'required|unique:products,code',
            'unit_price' => 'required|numeric',
            'category_id' => 'required',
            'supplier_id' => 'required',
            'qty' => 'required|integer',
            'margin_price' => 'required|numeric',
        ]);

        DB::beginTransaction();
        try {

            $product_path = null;
            if ($request->hasFile('product_image')) {
                $product_path = storeFile($request->file('product_image'), 'products');
            }

            $status = 1;
            if ($request->upi || $request->online || $request->cash) {
                $status = ($request->total_amount == ($request->upi + $request->online + $request->cash)) ? 2 : 3;
            }

            $product = ProductModel::create([
                'name' => $request->name,
                'code' => $request->productCode,
                'hsn_code' => $request->hsnCode,
                'category_id' => $request->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'description' => $request->description,
                'products_image' => $product_path,
                'branch_id' => $this->branch_id,
                'margin_price' => $request->margin_price,
                'qty' => $request->qty,
                'total_qty' => $request->qty
            ]);

            $product_id = $product->id;

            $productDetail = ProductDetailsModel::create([
                'product_id' => $product_id,
                'name' => $request->name,
                'code' => $request->productCode,
                'category_id' => $request->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'margin_price' => $request->margin_price,
                'description' => $request->description,
                'products_image' => $product_path,
                'branch_id' => $this->branch_id,
                'qty' => $request->qty,
                'total_amount' => $request->total_amount,
                'status' => $status
            ]);

            if (!empty($productDetail)) {
                if (!empty($request->upi)) {
                    ProductPaymentModel::create([
                        'product_dtl_id' => $productDetail->product_details_sid,
                        'payment_type' => 'upi',
                        'payment_date' => $request->pay_date,
                        'total_amount' => $request->total_amount,
                        'paid_amount' => $request->upi
                    ]);
                }

                if (!empty($request->online)) {
                    ProductPaymentModel::create([
                        'product_dtl_id' => $productDetail->product_details_sid,
                        'payment_type' => 'online',
                        'payment_date' => $request->pay_date,
                        'total_amount' => $request->total_amount,
                        'paid_amount' => $request->online
                    ]);
                }

                if (!empty($request->cash)) {
                    ProductPaymentModel::create([
                        'product_dtl_id' => $productDetail->product_details_sid,
                        'payment_type' => 'cash',
                        'payment_date' => $request->pay_date,
                        'total_amount' => $request->total_amount,
                        'paid_amount' => $request->cash
                    ]);
                }
            }

            ProductHistoryModel::create([
                'product_id' => $product_id,
                'branch_id' => $this->branch_id,
                'name' => $request->name,
                'code' => $request->productCode,
                'category_id' => $request->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'qty' => $request->qty,
                'description' => $request->description,
                'product_image' => $product_path,
                'action' => 'A',
                'created_by' => auth()->user()->id,
            ]);

            DB::commit();
            return redirect()->route('product-index')->with('success', 'Product added successfully!');
        } catch (\Exception $e) {
            DB::rollBack();

            \Log::info('While Adding Product' . $e->getMessage());

            return redirect()->back()->with('error', 'Something went wrong!')->withInput();
        }
    }

    public function edit($id)
    {
        $product = ProductModel::findOrFail($id);

        $categories = CategoryModel::where('branch_id', $this->branch_id)->get();

        $suppliers = SupplierModel::where('branch_id', $this->branch_id)->where('active_status', 1)->get();

        return view('products.edit', compact('product', 'categories', 'suppliers'));
    }

    public function purchase($id)
    {
        $product = ProductModel::findOrFail($id);

        $suppliers = SupplierModel::where('branch_id', $this->branch_id)->get();

        return view('products.purchase', compact('product', 'suppliers'));
    }

    public function savePurchase(Request $request)
    {
        $request->validate([
            'unit_price' => 'required|numeric',
            'margin_price' => 'required|numeric',
            'qty' => 'required|integer',
            'supplier_id' => 'required',
        ]);

        DB::beginTransaction();
        try {

            $product = ProductModel::findOrFail($request->product_id);

            if (($request->upi + $request->online + $request->cash) > $request->total_amount) {
                return redirect()->back()->with('error', 'Payment amount should not be greater than total amount')->withInput();
            }

            $product_path = $product->products_image;
            if ($request->hasFile('product_image')) {
                $product_path = storeFile($request->file('product_image'), 'products');
            }

            $status = 1;
            if ($request->upi || $request->online || $request->cash) {
                $status = ($request->total_amount == ($request->upi + $request->online + $request->cash)) ? 2 : 3;
            }

            $purchase = ProductDetailsModel::create([
                'product_id' => $request->product_id,
                'branch_id' => $this->branch_id,
                'name' => $product->name,
                'code' => $product->code,
                'category_id' => $product->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'margin_price' => $request->margin_price,
                'qty' => $request->qty,
                'status' => $status,
                'total_amount' => $request->total_amount,
                'description' => $request->description,
                'products_image' => $product_path,
            ]);


            $total_quantity = $product->qty + $purchase->qty;
            $product->update([
                'qty' => $total_quantity,
                'unit_price' => $purchase->unit_price,
                'margin_price' => $purchase->margin_price,
                'total_qty' => $product->total_qty + $purchase->qty,
            ]);

            if (!empty($purchase)) {
                if (!empty($request->upi)) {
                    ProductPaymentModel::create([
                        'product_dtl_id' => $purchase->product_details_sid,
                        'payment_type' => 'upi',
                        'payment_date' => $request->pay_date,
                        'total_amount' => $request->total_amount,
                        'paid_amount' => $request->upi
                    ]);
                }

                if (!empty($request->online)) {
                    ProductPaymentModel::create([
                        'product_dtl_id' => $purchase->product_details_sid,
                        'payment_type' => 'online',
                        'payment_date' => $request->pay_date,
                        'total_amount' => $request->total_amount,
                        'paid_amount' => $request->online
                    ]);
                }

                if (!empty($request->cash)) {
                    ProductPaymentModel::create([
                        'product_dtl_id' => $purchase->product_details_sid,
                        'payment_type' => 'cash',
                        'payment_date' => $request->pay_date,
                        'total_amount' => $request->total_amount,
                        'paid_amount' => $request->cash
                    ]);
                }
            }

            DB::commit();
            return redirect()->route('product-index')->with('success', 'Purchased products added successfully!');
        } catch (\Exception $e) {
            DB::rollBack();

            \Log::info('While adding Purchase for the product' . $e->getMessage());

            return redirect()->back()->with('error', 'Something went wrong!')->withInput();
        }
    }

    public function update(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'productCode' => 'required|unique:products,code,' . $request->product_id,
            'unit_price' => 'required|numeric',
            'category_id' => 'required',
            'supplier_id' => 'required',
            'qty' => 'required|integer',
            'margin_price' => 'required|numeric',
        ]);

        DB::beginTransaction();
        try {
            $product = ProductModel::findOrFail($request->product_id);

            // Handle product image update
            $product_path = $product->products_image;
            if ($request->hasFile('product_image')) {
                $product_path = storeFile($request->file('product_image'), 'products');

                if ($product->products_image && Storage::disk('s3')->exists($product->products_image)) {
                    Storage::disk('s3')->delete($product->products_image);
                }
            }

            $quantityDifference = $request->qty - $product->qty;
            $newTotalQuantity = $product->total_qty + $quantityDifference;

            if ($newTotalQuantity < 0) {
                throw new \Exception("Total quantity cannot be negative");
            }

            $product->update([
                'name' => $request->name,
                'code' => $request->productCode,
                'hsn_code' => $request->hsnCode,
                'category_id' => $request->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'margin_price' => $request->margin_price,
                'description' => $request->description,
                'products_image' => $product_path,
                'qty' => $request->qty,
                'total_qty' => $newTotalQuantity,
            ]);

            ProductHistoryModel::create([
                'product_id' => $request->product_id,
                'branch_id' => $this->branch_id,
                'name' => $request->name,
                'code' => $request->productCode,
                'category_id' => $request->category_id,
                'supplier_id' => $request->supplier_id,
                'unit_price' => $request->unit_price,
                'qty' => $request->qty,
                'description' => $request->description,
                'product_image' => $product_path,
                'action' => 'E',
                'created_by' => auth()->user()->id,
            ]);

            DB::commit();
            return redirect()->route('product-index')->with('success', 'Product updated successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('While Updating the product: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong: ' . $e->getMessage())->withInput();
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();

        try {
            $product = ProductModel::with(['details.payments'])
                ->where('active_flag', 'A')
                ->find($id);

            if (!$product) {
                return redirect()->back()->with('error', 'Product not found or already inactive!');
            }

            $inInvoice = InvoiceItemModel::where('product_id', $product->id)
                ->where('active_flag', 'A')
                ->exists();

            $inQuotes = QuotesItemsModel::where('product_id', $product->id)
                ->where('active_flag', 'A')
                ->exists();

            $inPayment = ProductPaymentModel::whereIn('product_dtl_id', $product->details->pluck('product_details_sid'))
                ->where('active_flag', 'A')
                ->exists();

            if ($inInvoice || $inQuotes || $inPayment) {
                DB::rollBack();
                return redirect()->back()->with('error', 'Cannot delete product: It exists in active invoices, quotes, or payments!');
            }

            // Create history record
            ProductHistoryModel::create([
                'product_id' => $product->id,
                'branch_id' => $this->branch_id,
                'name' => $product->name,
                'code' => $product->code,
                'category_id' => $product->category_id,
                'supplier_id' => $product->supplier_id,
                'unit_price' => $product->unit_price,
                'qty' => $product->qty,
                'description' => $product->description,
                'product_image' => $product->products_image,
                'action' => 'D',
                'created_by' => auth()->id(),
            ]);

            foreach ($product->details as $detail) {
                if ($detail->payments->isNotEmpty()) {
                    $detail->payments->delete();
                }
                $detail->delete();
            }

            $product->delete();

            DB::commit();

            return redirect()->route('product-index')->with('success', 'Product deleted successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error("Product deactivation failed for ID {$id}: " . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete product');
        }
    }
}