<?php

namespace App\Http\Controllers;

use App\Models\BranchModel;
use App\Models\InvoiceModel;
use App\Models\QuotesModel;
use App\Models\StandardPaperSizeModel;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Log;
use Mpdf\Mpdf;
use Mpdf\Config\ConfigVariables;
use Mpdf\Config\FontVariables;


class PDFController extends Controller
{


    public function generateInvoice(Request $request)
    {
        try {
            // Retrieve invoice data (same as your existing code)
            $addedInvoice = InvoiceModel::with('client.user', 'branch.company', 'invoiceItems.invoiceItemTaxes', 'payments.mode', 'invoiceItems.products.category.unitOfMass')
                ->where('id', $request->invoice_id)
                ->first();

            if (!$addedInvoice) {
                return response()->json(['error' => 'Invoice not found'], 404);
            }

            // Process invoice items (same as your existing code)
            $invoiceItems = $addedInvoice->invoiceItems;
            $products = [];
            $discount = 0;
            $total = 0;

            foreach ($invoiceItems as $item) {
                $gst = $item->invoiceItemTaxes->sum('tax');
                $subtotal = ($item->price * $gst) / 100;
                $totalWithoutGst = $item->quantity * $item->price;
                $totalWithGst = ($item->quantity * $subtotal) + $totalWithoutGst;
                $total += $totalWithGst;
                $products[] = [
                    'name' => $item->products->name ?? '',
                    'code' => $item->products->code ?? '',
                    'quantity' => $item->quantity ?? '',
                    'uom' => $item->products->category->unitOfMass->name ?? '',
                    'unit_price' => $item->price ?? '',
                    'gst' => $item->invoiceItemTaxes->sum('tax') ?? 0,
                    'total_price' => $item->quantity * $subtotal,
                ];
            }

            // Calculate discount (same as your existing code)
            if ($addedInvoice->discount_type == 1) {
                $discount = $addedInvoice->discount;
            } elseif ($addedInvoice->discount_type == 2) {
                $discount = ($total * $addedInvoice->discount) / 100;
            } else {
                $discount = 0;
            }

            $sumQty = 0;
            foreach ($addedInvoice->invoiceItems as $item) {
                $sumQty += $item->quantity;
            }

            // Prepare data for view
            $data = [
                'companyName' => $addedInvoice->branch->company->name,
                'companyAddress' => $addedInvoice->branch->company->address,
                'contact_no' => $addedInvoice->branch->company->phone,
                'email' => $addedInvoice->branch->company->email,
                'clientName' => $addedInvoice->client->user->name,
                'clientAddress' => $addedInvoice->client->user->location,
                'gst_no' => $addedInvoice->branch->gst_no,
                'invoiceNumber' => $addedInvoice->invoice_id,
                'invoiceDate' => $addedInvoice->invoice_date ? \Carbon\Carbon::parse($addedInvoice->invoice_date)->format('d-m-Y') : '',
                'dueDate' => $addedInvoice->due_date ? \Carbon\Carbon::parse($addedInvoice->due_date)->format('d-m-Y') : '',
                'discount' => $discount,
                'products' => $products,
                'type' => 'I',
                'payment_terms' => $addedInvoice->payments->isNotEmpty() && $addedInvoice->payments->first()->mode
                    ? $addedInvoice->payments->first()->mode->name
                    : 'N/A',
                'totalQty' => $sumQty,
                'terms' => isset($addedInvoice->branch) ? $addedInvoice->branch->declaration : '',
            ];

            // Render the HTML
            $html = view('pdf.others', $data)->render();

            // Configure mPDF for Tamil support
            $defaultConfig = (new ConfigVariables())->getDefaults();
            $fontDirs = $defaultConfig['fontDir'];

            $defaultFontConfig = (new FontVariables())->getDefaults();
            $fontData = $defaultFontConfig['fontdata'];

            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            $mpdf->WriteHTML($html);

            if ($request->type == 'download') {
                return response($mpdf->Output('invoice.pdf', 'S'))
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'attachment; filename="' . $addedInvoice->invoice_id . '.pdf"');
            }

            if ($request->type == 'print') {
                return response($mpdf->Output('invoice.pdf', 'S'))
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'inline; filename="' . $addedInvoice->invoice_id . '.pdf"');
            }

            return response($mpdf->Output('invoice.pdf', 'S'))
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $addedInvoice->invoice_id . '.pdf"');

        } catch (\Exception $e) {
            Log::error('Error generating invoice: ' . $e->getMessage());
            return response()->json(['error' => 'Something went wrong. Please try again later.'], 500);
        }
    }
    public function generateQuote(Request $request)
    {
        try {

            $addedQuote = QuotesModel::with('client.user', 'branch.company', 'quotesItem.quoteItemTaxes', 'quotesItem.product.category.unitOfMass')
                ->where('id', $request->quote_id)
                ->first();

            if (!$addedQuote) {
                return response()->json(['error' => 'Invoice not found'], 404);
            }

            $quoteItems = $addedQuote->quotesItem;
            $products = [];
            $discount = 0;
            $total = 0;

            foreach ($quoteItems as $item) {
                $gst = $item->quoteItemTaxes->sum('tax');
                $subtotal = ($item->price * $gst) / 100;
                $totalWithoutGst = $item->quantity * $item->price;
                $totalWithGst = ($item->quantity * $subtotal) + $totalWithoutGst;
                $total += $totalWithGst;
                $products[] = [
                    'name' => $item->product->name ?? '',
                    'code' => $item->product->code ?? '',
                    'quantity' => $item->quantity ?? '',
                    'uom' => $item->product->category->unitOfMass->name ?? '',
                    'unit_price' => $item->price ?? '',
                    // 'gst' => $item->quoteItemTaxes->sum('tax') ?? 0,
                    'total_price' => $item->quantity * $subtotal,
                ];
            }

            // Calculate the discount
            if ($addedQuote->discount_type == 1) {
                $discount = $addedQuote->discount;
            } elseif ($addedQuote->discount_type == 2) {
                $discount = ($total * $addedQuote->discount) / 100;
            } else {
                $discount = 0;
            }

            $data = [
                'companyName' => $addedQuote->branch->company->name,
                'companyAddress' => $addedQuote->branch->company->address,
                'contact_no' => $addedQuote->branch->company->phone,
                'email' => $addedQuote->branch->company->email,
                'clientName' => $addedQuote->client->user->name,
                'clientAddress' => $addedQuote->client->user->location,
                'gst_no' => $addedQuote->branch->gst_no,
                'invoiceNumber' => $addedQuote->quote_id,
                'invoiceDate' => $addedQuote->quote_date ? \Carbon\Carbon::parse($addedQuote->quote_date)->format('d-m-Y') : '',
                'dueDate' => $addedQuote->due_date ? \Carbon\Carbon::parse($addedQuote->due_date)->format('d-m-Y') : '',
                'discount' => $discount,
                'products' => $products,
                'type' => 'Q',
            ];

            // Check if the template exists
            if (!view()->exists('pdf.others')) {
                return response()->json(['error' => 'Invoice template not found'], 404);
            }

            // Generate the PDF
            $pdf = Pdf::loadView('pdf.others', $data);
            ob_end_clean();

            if ($request->type == 'download') {
                return response($pdf->output())
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'attachment; filename="' . $addedQuote->quote_id . '.pdf"');
            }

            if ($request->type == 'print') {
                return response($pdf->output())
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'inline; filename="' . $addedQuote->quote_id . '.pdf"');
            }

            // Default: download the PDF if no type is provided
            return response($pdf->output())
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="' . $addedQuote->quote_id . '.pdf"');


        } catch (\Exception $e) {

            Log::info('Error in Quote generate' . $e->getMessage());

            return response()->json(['error' => 'Something went wrong. Please try again later.'], 500);
        }
    }

    public function openQoutePdfPage(Request $request, $id)
    {
        $addedQuote = QuotesModel::with('client.user', 'branch.company', 'quotesItem.quoteItemTaxes', 'quotesItem.product.category.unitOfMass')
            ->where('id', $id)
            ->first();
        $settings = BranchModel::where('id', $addedQuote->branch_id)->first();
        $paper = StandardPaperSizeModel::where('oprntl_flag', 'A')->where('view_flag', 'A')->get();
        $paper_size = $paper->where('id', $request->paperId)->first();

        if (isset($paper_size) ? $paper_size->name == 'Others' : $settings->paper->name == 'Others') {

            return view('quotes.pdf.quote-generate-pdf', [
                'addedQuote' => $addedQuote,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'Thermal Printer' : $settings->paper->name == 'Thermal Printer') {

            return view('quotes.pdf.pdf', [
                'addedInvoice' => $addedQuote,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'A4' : $settings->paper->name == 'A4') {

            return view('quotes.pdf.a4Pdf', [
                'addedInvoice' => $addedQuote,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'A3' : $settings->paper->name == 'A3') {

            return view('quotes.pdf.a3Pdf', [
                'addedInvoice' => $addedQuote,
                'settings' => $settings,
                'paper' => $paper
            ]);
        }

    }

    public function generatePdf(Request $request, $id)
    {
        // dd($request->all());
        $addedInvoice = InvoiceModel::with('client.user', 'branch.company', 'invoiceItems.invoiceItemTaxes', 'payments.mode', 'invoiceItems.products.category.unitOfMass')
            ->where('id', $id)
            ->first();

        $settings = BranchModel::with('paper')->where('id', $addedInvoice->branch_id)->first();

        $paper = StandardPaperSizeModel::where('oprntl_flag', 'A')->where('view_flag', 'A')->get();

        $paper_size = $paper->where('id', $request->paperId)->first();
        // dd($paper_size);
        if (isset($paper_size) ? $paper_size->name == 'Others' : $settings->paper->name == 'Others') {

            return view('pdf.billGenerate', [
                'addedInvoice' => $addedInvoice,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'Thermal Printer' : $settings->paper->name == 'Thermal Printer') {

            return view('pdf.pdf', [
                'addedInvoice' => $addedInvoice,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'A4' : $settings->paper->name == 'A4') {

            return view('pdf.a4Pdf', [
                'addedInvoice' => $addedInvoice,
                'settings' => $settings,
                'paper' => $paper
            ]);
        } elseif (isset($paper_size) ? $paper_size->name == 'A3' : $settings->paper->name == 'A3') {

            return view('pdf.a3Pdf', [
                'addedInvoice' => $addedInvoice,
                'settings' => $settings,
                'paper' => $paper
            ]);
        }

    }

}