<?php

namespace App\Imports;

use App\Models\TaxesModel;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\RemembersRowNumber;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\BeforeImport;
use Maatwebsite\Excel\Events\AfterImport;

class TaxImport implements ToModel, WithHeadingRow, WithEvents
{
    use RemembersRowNumber;
    protected $row;

    public function registerEvents(): array
    {
        return [
            BeforeImport::class => function (BeforeImport $event) {
                DB::beginTransaction();
            },
            AfterImport::class => function (AfterImport $event) {
                $errors = session()->get('tax_err', []);
                if (empty($errors)) {
                    DB::commit();
                } else {
                    DB::rollBack();
                }
            }
        ];
    }

    public function model(array $row)
    {
        $this->row = $this->getRowNumber();
        $hasErrors = false;

        // Validate required fields
        if (empty($row['tax_name'])) {
            $this->storeErrors('Tax Name', $row['tax_name'] ?? '', 'Enter Tax Name');
            $hasErrors = true;
        }

        if (empty($row['tax_value'])) {
            $this->storeErrors('Tax Value', $row['tax_value'] ?? '', 'Enter Tax Value');
            $hasErrors = true;
        }

        if (!isset($row['is_active'])) {
            $this->storeErrors('Active Tax', $row['is_active'] ?? '', 'Select Is Active');
            $hasErrors = true;
        }

        if (!isset($row['is_default'])) {
            $this->storeErrors('Default Tax', $row['is_default'] ?? '', 'Select Is Default');
            $hasErrors = true;
        }

        if ($hasErrors) {
            return null;
        }

        try {
            return new TaxesModel([
                'name' => $row['tax_name'],
                'value' => $row['tax_value'],
                'is_default' => $row['is_default'] == true ? 1 : 0,
                'show_flg' => $row['is_active'] == true ? 'Y' : 'N',
                'branch_id' => session('branch_id'),
            ]);
        } catch (\Exception $e) {
            // $this->storeErrors('System Error', '', $e->getMessage());
            \Log::error('While importing tax at row ' . $this->row . ': ' . $e->getMessage());
            return null;
        }
    }

    private function storeErrors($key, $value = ' ', $error = 'Please check the spelling')
    {
        $errors = session()->get('tax_err', []);
        $errors[] = htmlspecialchars($key) . ': ' . htmlspecialchars($value) . ' ' . htmlspecialchars($error) . ' at row no: ' . $this->row;
        session()->put('tax_err', $errors);
    }
}