<?php

namespace App\Imports;

use App\Models\CategoryModel;
use App\Models\UomModel;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\RemembersRowNumber;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\BeforeImport;
use Maatwebsite\Excel\Events\AfterImport;

class CategoriesImport implements ToModel, WithHeadingRow, WithEvents
{
    use RemembersRowNumber;
    protected $row;

    public function registerEvents(): array
    {
        return [
            BeforeImport::class => function (BeforeImport $event) {
                DB::beginTransaction();
            },
            AfterImport::class => function (AfterImport $event) {
                $errors = session()->get('cat_err', []);
                if (empty($errors)) {
                    DB::commit();
                } else {
                    DB::rollBack();
                }
            }
        ];
    }

    public function model(array $row)
    {
        $this->row = $this->getRowNumber();
        $hasErrors = false;

        // Validate required fields
        if (empty($row['categoryname'])) {
            $this->storeErrors('Category', $row['categoryname'] ?? '', 'Enter a Category Name');
            $hasErrors = true;
        }

        // Check if category already exists
        $category = CategoryModel::where('name', $row['categoryname'])->first();
        if ($category) {
            $this->storeErrors('Category', $row['categoryname'], 'Already Exists');
            $hasErrors = true;
        }

        // Validate UOM
        $uomId = $this->getUnit($row['unitofmeasurement'] ?? '');
        if ($uomId === null) {
            $hasErrors = true;
        }

        if ($hasErrors) {
            return null;
        }

        try {
            return new CategoryModel([
                'name' => $row['categoryname'],
                'uom' => $uomId,
                'branch_id' => session('branch_id'),
            ]);
        } catch (\Exception $e) {
            \Log::error('While importing category at row ' . $this->row . ': ' . $e->getMessage());
            $this->storeErrors('System Error', '', $e->getMessage());
            return null;
        }
    }

    protected function getUnit($data)
    {
        if (empty($data)) {
            $this->storeErrors('Unit of Measurement', $data, 'Please enter a unit');
            return null;
        }

        $uom = UomModel::where('name', $data)->pluck('id')->first();
        if (empty($uom)) {
            $this->storeErrors('Unit of Measurement', $data, 'Please check the spelling');
            return null;
        }

        return $uom;
    }

    private function storeErrors($key, $value = ' ', $error = 'Please check the spelling')
    {
        $errors = session()->get('cat_err', []);
        $errors[] = htmlspecialchars($key) . ': ' . htmlspecialchars($value) . ' ' . htmlspecialchars($error) . ' at row no: ' . $this->row;
        session()->put('cat_err', $errors);
    }
}