<?php

namespace App\Http\Controllers;
use App\Mail\Ordercreated;
use App\Models\LoyaltyTransactionModel;
use App\Models\OrdersItemsModel;
use App\Models\PaymentsModel;
use App\Models\PaymentStatusModel;
use App\Models\RightsModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Mpdf\Mpdf;
use App\Mail\InvoiceCreated;
use App\Models\BranchModel;
use App\Models\ClientAddressModel;
use App\Models\ClientModel;
use App\Models\InvoiceItemModel;
use App\Models\InvoiceItemTaxModel;
use App\Models\InvoiceModel;
use App\Models\ProductModel;
use App\Models\OrdersModel;
use App\Models\TaxesModel;
use App\Models\User;

class OrdersController extends Controller
{
    public function index(Request $request)
    {
        if (Auth::user()->user_flg == 'C') {
            $client_id = ClientModel::where('user_id', Auth::user()->id)->pluck('id')->first();
            $orders = OrdersModel::with('client.user', 'branch')->where('branch_id', session('branch_id'))->where('client_id', $client_id)->orderBy('id', 'desc')->get();
        } else {
            $orders = OrdersModel::with('client.user')->where('branch_id', session('branch_id'))->orderBy('id', 'desc')->get();
        }
        return view('orders.index', compact('orders'));
    }
    public function add()
    {
        $clients = ClientModel::with('user', 'clientAddress')
            ->when(Auth::user()->user_flg == 'C', function ($query) {
                $query->where('user_id', Auth::id());
            })->get();

        $products = ProductModel::with('branch')->where('branch_id', session('branch_id'))->whereIn('product_type', [2, 3])->get();

        return view('orders.add', compact('clients', 'products'));
    }

    public function store(Request $request)
    {

        $branch_id = session('branch_id');
        $request->validate([
            'dynamicFields.*.qty' => 'required|numeric|lte:dynamicFields.*.available_qty',
            'dynamicFields.*.unit_price' => 'required|numeric',
            'dynamicFields.*.amount' => 'required|numeric',
            'client_num' => 'required',
        ]);
        DB::beginTransaction();
        try {
            $user = User::where('phone', $request->client_num)->first();

            $client_id = saveClient($request);

            if (!$client_id) {
                return redirect()->back()->with('error', 'Failed to create client');
            }

            $lastorder = OrdersModel::orderBy('id', 'desc')->first();
            $order_id = $this->generateOrderId($lastorder);

            $order = OrdersModel::create([
                'order_id' => $order_id,
                'client_id' => $client_id,
                'order_date' => $request->order_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type ?? null,
                'discount' => $request->discount ?? null,
                'note' => $request->note,
                'term' => $request->term,
                'branch_id' => $branch_id,
                'status' => $request->status ?? '',
                'client_address_id' => $request->selected_address,
            ]);

            foreach ($request->products as $field) {
                $product = ProductModel::find($field['id']);

                OrdersItemsModel::create([
                    'order_id' => $order->id,
                    'product_id' => $field['id'],
                    'product_name' => $product->name ?? '',
                    'available_quantity' => $product->qty,
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'total' => $field['amount'],
                ]);
            }

            if ($request->filled('selected_address') && $client_id) {
                ClientAddressModel::where('client_id', $client_id)->update(['oprtnl_flag' => 'I']);

                ClientAddressModel::where('id', $request->selected_address)
                    ->where('client_id', $client_id)
                    ->update(['oprtnl_flag' => 'A']);
            }

            $settings = BranchModel::where('id', $order->branch_id)->first();

            $orderDetails = OrdersModel::with('client.user', 'clientAddress', 'ordersItem.product')->where('id', $order->id)->where('branch_id', session('branch_id'))->first();
            $data = [
                'addedOrder' => $orderDetails,
                'settings' => $settings
            ];


            $html = view('orders.email-pdf.order-generate-pdf_email', $data)->render();
            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            $mpdf->WriteHTML($html);
            $pdfContent = $mpdf->Output('', 'S');
            $admin_mail = User::where('user_flg', 'A')->where('branch_id', session('branch_id'))->pluck('email')->toArray();

            Mail::to($admin_mail)->send(new Ordercreated($pdfContent, $order, $settings));



            DB::commit();

            return redirect()->route('order-pdf', ['id' => $order->id])->with('success', 'Order created successfully');

        } catch (\Exception $e) {

            DB::rollBack();

            \Log::info('While Add  Order' . $e->getMessage());

            return redirect()->back()->with('error', 'Failed to add order: ' . $e->getMessage())->withInput();
        }
    }


    public function edit($id, $type)
    {
        $branch_id = session('branch_id');

        $clients = ClientModel::with('user', 'clientAddress')->get();

        $order = OrdersModel::with('ordersItem', 'client.user', 'client.clientAddress')->findOrFail($id);

        $discount_types = DB::table('discount_type')->get();

        $products = ProductModel::where('branch_id', $branch_id)->whereIn('product_type', [2, 3])->get();

        $branch = BranchModel::where('id', session('branch_id'))->first();

        $default_taxes = TaxesModel::where('is_default', 1)->where('branch_id', session('branch_id'))->pluck('id')->toArray();

        $all_taxes = TaxesModel::where('show_flg', 'Y')->where('branch_id', session('branch_id'))->get();

        if ($type == 'edit') {
            return view('orders.edit', compact('clients', 'products', 'order', 'branch', 'type'));
        } else {
            return view('orders.convert', compact('clients', 'products', 'order', 'type', 'all_taxes', 'branch', 'discount_types', 'default_taxes'));
        }
    }

    public function update(Request $request)
    {

        if ($request->type == 'convert') {
            return $this->processConvert($request);
        }

        try {
            DB::beginTransaction();

            $order = OrdersModel::findOrFail($request->id);

            $order->update([
                'client_id' => $request->client_id,
                'order_date' => $request->order_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type ?? null,
                'discount' => $request->discount ?? null,
                'client_address_id' => $request->selected_address,
            ]);

            $productIds = [];

            foreach ($request->products as $field) {
                $product = ProductModel::findOrFail($field['id']);

                $ordersItem = OrdersItemsModel::updateOrCreate(
                    [
                        'order_id' => $order->id,
                        'product_id' => $field['id'],
                    ],
                    [
                        'product_name' => $product->name,
                        'available_quantity' => $product->qty,
                        'quantity' => $field['qty'],
                        'price' => $field['unit_price'],
                        'total' => $field['amount'],
                    ]
                );

                $productIds[] = $field['id'];
            }

            OrdersItemsModel::where('order_id', $order->id)
                ->whereNotIn('product_id', $productIds)
                ->delete();

            if ($request->filled('selected_address') && $request->client_id) {
                ClientAddressModel::where('client_id', $request->client_id)->update(['oprtnl_flag' => 'I']);

                ClientAddressModel::where('id', $request->selected_address)
                    ->where('client_id', $request->client_id)
                    ->update(['oprtnl_flag' => 'A']);
            }


            DB::commit();

            return redirect()->route('orders-index')
                ->with('success', 'Order updated successfully');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('While updating Order: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to update order: ' . $e->getMessage())->withInput();
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            $order = OrdersModel::with('ordersItem')->findOrFail($id);
            $order->ordersItem()->delete();
            $order->delete();

            DB::commit();

            return redirect()->route('orders-index')->with('success', 'Order deleted successfully');

        } catch (\Exception $e) {

            DB::rollBack();

            \Log::info('While Deleting Order' . $e->getMessage());

            return redirect()->route('orders-index')->with('error', 'An error occurred while deleting the Order');
        }
    }

    public function processConvert($request)
    {
        $request->validate([
            'dynamicFields.*.qty' => 'required|numeric|lte:dynamicFields.*.available_qty',
            'dynamicFields.*.unit_price' => 'required|numeric',
            'dynamicFields.*.amount' => 'required|numeric',
        ]);

        if (($request->cash + $request->upi + $request->online) > $request->total_amount) {
            return back()->with('error', 'The total of cash, UPI, and online payments cannot exceed the total amount.');
        }

        try {
            DB::beginTransaction();

            $order = OrdersModel::findOrFail($request->id);            
            $invoice_id = getPrefixes($request, $order->client_id);

            $loyalty = BranchModel::where('id', session('branch_id'))->first();

            $clientPoints = ClientModel::where('id', $order->client_id)->first();

            $loyalty_amount = $loyalty->loyalty_amount;
            $loyalty_config = $loyalty->loyalty_config;

            if (!empty($request->redeem_points)) {
                $current_point = $request->available_point - $request->redeem_points;

                LoyaltyTransactionModel::create([
                    'client_id' => $order->client_id,
                    'points' => $request->redeem_points,
                    'type' => 'redeem',
                    'reference_invoice' => $invoice_id,
                    'created_by' => Auth::user()->id,
                ]);
                $clientPoints->update([
                    'loyalty_points' => $current_point,
                ]);

            }

            if ($loyalty_config == 'A') {
                $earned_points = floor($request->total_amount / $loyalty_amount);
                $points = $clientPoints->loyalty_points + $earned_points;
                $clientPoints->update([
                    'loyalty_points' => $points,
                ]);

                LoyaltyTransactionModel::create([
                    'client_id' => $order->client_id,
                    'points' => $earned_points,
                    'type' => 'earn',
                    'reference_invoice' => $invoice_id,
                    'created_by' => Auth::user()->id,
                ]);

            }

            if ($request->selected_address) {
                $order = OrdersModel::findOrFail($request->id);

                $order->update([
                    'client_address_id' => $request->selected_address,
                ]);
            }
            $paid_amount = $request->upi + $request->cash + $request->online;
            if ($paid_amount == 0) {
                $status = 'Unpaid';
            } elseif ($paid_amount == $request->total_amount) {
                $status = 'Paid';
            } else {
                $status = 'Partially Paid';
            }
            $status_id = PaymentStatusModel::where('name', $status)->value('id');

            $invoice = InvoiceModel::create([
                'invoice_id' => $invoice_id,
                'client_id' => $order->client_id,
                'invoice_date' => $request->order_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type,
                'discount' => $request->discount,
                'note' => $request->note,
                'term' => $request->term,
                'template_id' => $request->template_id,
                'branch_id' => session('branch_id'),
                'gst' => $request->gst,
                'order_id' => $order->order_id,
                'status' => $status_id,
                'created_by' => Auth::user()->id,
            ]);

            //  $products = [];

            foreach ($request->products as $field) {
                $product = ProductModel::findOrFail($field['id']);
                $available_qty = $product->qty - $field['qty'];

                $product->update(['qty' => $available_qty]);

                $item = InvoiceItemModel::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $field['id'],
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'margin_price' => $product->margin_price,
                    'total' => $field['amount'],
                    'product_name' => $product->name,
                ]);
                if ($request->gst == "Y") {
                    foreach ($field['tax'] as $taxId) {
                        InvoiceItemTaxModel::create([
                            'invoice_item_id' => $item->id,
                            'tax_id' => $taxId,
                            'tax' => TaxesModel::find($taxId)->value,
                        ]);
                    }
                }
            }

            if ($status_id == '2' || $status_id == '3') {
                if (!empty($request->upi)) {
                    PaymentsModel::create([
                        'invoice_id' => $invoice->id,
                        'amount' => $request->upi,
                        'total_amount' => $request->total_amount,
                        'payment_mode' => 1
                    ]);
                }
                if (!empty($request->cash)) {
                    PaymentsModel::create([
                        'invoice_id' => $invoice->id,
                        'amount' => $request->cash,
                        'total_amount' => $request->total_amount,
                        'payment_mode' => 3
                    ]);
                }
                if (!empty($request->online)) {
                    PaymentsModel::create([
                        'invoice_id' => $invoice->id,
                        'amount' => $request->online,
                        'total_amount' => $request->total_amount,
                        'payment_mode' => 2
                    ]);
                }
            }

            $order->update([
                'oprntl_flag' => 'C',
                'status' => $request->status,
            ]);

            if ($request->filled('selected_address') && $order->client_id) {
                ClientAddressModel::where('client_id', $order->client_id)->update(['oprtnl_flag' => 'I']);

                ClientAddressModel::where('id', $request->selected_address)
                    ->where('client_id', $order->client_id)
                    ->update(['oprtnl_flag' => 'A']);
            }

            $settings = BranchModel::where('id', $invoice->branch_id)->first();

            $invoiceDetails = InvoiceModel::with([
                'invoiceItems.invoiceItemTaxes',
                'invoiceItems.products.category.unitOfMass',
                'order.ordersItem.product.category.unitOfMass',
                'order.clientAddress',
                'branch.company',
                'client.user',
            ])->where('order_id', $order->order_id)->where('invoice_id', $invoice->invoice_id)->first();
            $redeem = LoyaltyTransactionModel::where('client_id', $invoice->client_id)->where('reference_invoice', $invoice->invoice_id)->get();

            $redeemed_points = $redeem->where('type', 'redeem')->sum('points');

            $redeemed_value = $redeemed_points * $invoice->branch->point_redemption;

            $data = [
                'addedInvoice' => $invoiceDetails,
                'settings' => $settings,
                'redeemed_points' => $redeemed_points ?? 0,
                'redeemed_value' => $redeemed_value ?? 0
            ];

            $html = view('pdf.billGenerate_email', $data)->render();
            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            try {
                $mpdf->WriteHTML($html);
                $pdfContent = $mpdf->Output('', 'S');
                $user = User::find(ClientModel::find($order->client_id)?->user_id);

                Mail::to($user->email)->send(new InvoiceCreated($pdfContent, $invoice, $settings));
            } catch (\Mpdf\MpdfException $e) {
                \Log::error('PDF generation error' . $e->getMessage());
            }
            DB::commit();

            return redirect()->route('orders-index')->with('success', 'Order converted to invoice successfully');

        } catch (\Exception $e) {
            DB::rollBack();
            // dd($e);
            \Log::error('While converting Order to Invoice: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to convert order: ' . $e->getMessage())->withInput();
        }
    }

    protected function generateOrderId($lastorder)
    {
        if ($lastorder) {
            $lastOrderId = $lastorder->order_id;
            $parts = explode('-', $lastOrderId);
            $prefix = $parts[0];
            $number = (int) $parts[1];
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }
        return 'OSORD-001';
    }   
}
