<?php

namespace App\Http\Controllers;

use App\Models\CentralizedUsers;
use App\Models\CityModel;
use App\Models\ClientAddressModel;
use App\Models\ClientModel;
use App\Models\CountryModel;
use App\Models\InvoiceModel;
use App\Models\OrdersModel;
use App\Models\QuotesModel;
use App\Models\RightsModel;
use App\Models\StateModel;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class ClientController extends Controller
{
    protected $branch_id;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->branch_id = session('branch_id');
            return $next($request);
        });
    }

    public function index(Request $request)
    {
        if (Auth::user()->user_flg == 'C') {
            $clients = ClientModel::with('user')->where('branch_id', session('branch_id'))->where('user_id', Auth::user()->id)->get();
        } else {
            $clients = ClientModel::with('user')->orderBy('id', 'desc')->get();
        }
        return view('clients.index', compact('clients'));
    }

    public function add()
    {
        $countries = CountryModel::get();
        return view('clients.add', compact('countries'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            // 'email' => 'required|email|unique:users,email',
            'email' => [
                'required',
                'email',
                'unique:users,email',
                Rule::unique('second_db.users', 'email')
            ],
            'contact_no' => 'required|string|max:15|unique:users,phone',
            'password' => 'required|string|min:6|confirmed',
            'postal_code' => 'required|string|max:10',
            'country' => 'required|max:255',
            'state' => 'required|max:255',
            'city' => 'required|max:255',
            'company_name' => 'required|string|max:255',
            'address' => 'required|string|max:255',
            'notes' => 'nullable|string',
            'gstin' => 'nullable|string|max:15',
            'profile_image' => 'nullable|image',
            'active_status' => 'required',
        ]);

        DB::beginTransaction();
        try {

            $right = RightsModel::where('rights_type', 'client')->first();
            $clientRight = $right ? $right->id : null;

            $user = User::create([
                'name' => $validated['first_name'],
                'last_name' => $validated['last_name'],
                'email' => $validated['email'],
                'phone' => $validated['contact_no'],
                'password' => Hash::make($validated['password']),
                'user_flg' => 'C',
                'active_status' => $validated['active_status'],
                'branch_id' => $this->branch_id,
                'rights_id' => $clientRight
            ]);

            $profile_image_path = null;
            if ($request->hasFile('profile_image')) {
                $profile_image_path = storeFile($request->file('profile_image'), 'client');
            }

            $client = ClientModel::create([
                'user_id' => $user->id,
                'profile_img' => $profile_image_path,
                'website' => $request->input('website'),
                'postal_code' => $validated['postal_code'],
                'country_id' => $validated['country'],
                'state_id' => $validated['state'],
                'city_id' => $validated['city'],
                'company_name' => $validated['company_name'],
                'address' => $validated['address'],
                'note' => $validated['notes'],
                'gstin' => $validated['gstin'],
                'branch_id' => $this->branch_id,
                'active_status' => $validated['active_status'],
            ]);
            DB::table('company_user')->insert([
                'user_id' => $user->id,
                'company_id' => 1,
            ]);
            $allBranchIds = DB::table('branch')->pluck('id');
            $allCompanyIds = DB::table('company')->pluck('id');

            $insertData = [];
            $comapanyInsertData = [];

            foreach ($allBranchIds as $branchId) {
                $insertData[] = [
                    'user_id' => $user->id,
                    'branch_id' => $branchId,
                ];
            }
            foreach ($allCompanyIds as $companyId) {
                $comapanyInsertData[] = [
                    'user_id' => $user->id,
                    'company_id' => $companyId,
                ];
            }
            DB::table('branch_user')->insert($insertData);
            DB::table('company_user')->insert($comapanyInsertData);
            ClientAddressModel::create([
                'branch_id' => $this->branch_id,
                'client_id' => $client->id,
                'address' => $validated['address'],
                'created_by' => $client->id,
            ]);

            centralizedUsers($validated['first_name'] . ' ' . $validated['last_name'], $validated['email'], $validated['contact_no']);

            DB::commit();
            return redirect()->route('client-index')->with('success', 'Client added successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Client Add' . $e);
            return back()->with('error', 'Something went wrong!')->withInput();
        }
    }

    public function edit($id)
    {
        $client = ClientModel::with('user')->findOrFail($id);

        $countries = CountryModel::get();
        $states = StateModel::where('country_id', $client->country_id)->get();
        $cities = CityModel::where('state_id', $client->state_id)->get();

        return view('clients.edit', compact('client', 'countries', 'states', 'cities'));
    }

    public function update(Request $request)
    {
        $client = ClientModel::with('user')->findOrFail($request->client_id);

        $rules = [
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            // 'email' => 'required|email|unique:users,email,' . $client->user->id,
            'email' => [
                'required',
                'email',
                'unique:users,email,' . $client->user->id,
                Rule::unique('second_db.users', 'email')->ignore($client->user->email, 'email'),
            ],
            'contact_no' => 'required|string|unique:users,phone,' . $client->user->id,
            'postal_code' => 'required|string|max:10',
            'country' => 'required|max:255',
            'state' => 'required|max:255',
            'city' => 'required|max:255',
            'company_name' => 'required|string|max:255',
            'address' => 'required|string|max:255',
            'notes' => 'nullable|string',
            'gstin' => 'nullable|string|max:15',
            'profile_image' => 'nullable|image',
            'active_status' => 'required',
        ];

        $validated = $request->validate($rules);

        DB::beginTransaction();

        try {
            $client->country_id = $validated['country'];
            $client->state_id = $validated['state'];
            $client->city_id = $validated['city'];
            $client->website = $request->input('website');
            $client->postal_code = $validated['postal_code'];
            $client->company_name = $validated['company_name'];
            $client->address = $validated['address'];
            $client->note = $validated['notes'];
            $client->gstin = $validated['gstin'];
            $client->active_status = $validated['active_status'];

            $client->user->name = $validated['first_name'];
            $client->user->last_name = $validated['last_name'];
            $client->user->email = $validated['email'];
            $client->user->phone = $validated['contact_no'];
            $client->user->active_status = $validated['active_status'];
            $client->user->save();

            if ($request->hasFile('profile_image')) {
                // Delete old image if exists
                if ($client->profile_img) {
                    Storage::delete($client->profile_img);
                }
                $client->profile_img = storeFile($request->file('profile_image'), 'client');
            }

            $client->save();

            DB::commit();
            if (Auth::user()->user_flg == 'C') {
                return redirect()->route('orders-index')->with('success', 'Client updated successfully!');
            }
            return redirect()->route('client-index')->with('success', 'Client updated successfully!');
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Client Update' . $e);
            return back()->with('error', 'Client not found!')->withInput();
        }
    }

    public function delete($id)
    {
        $client = ClientModel::find($id);
        if ($client) {

            $messageParts = [];

            $invoice = InvoiceModel::where('client_id', $client->id)->where('active_flag', 'A')->first();
            if ($invoice) {
                $messageParts[] = 'invoices';
            }

            $quotes = QuotesModel::where('client_id', $client->id)->where('active_flag', 'A')->first();
            if ($quotes) {
                $messageParts[] = 'quotes';
            }

            $order = OrdersModel::where('client_id', $client->id)->where('active_flag', 'A')->first();
            if ($order) {
                $messageParts[] = 'orders';
            }

            if (!empty($messageParts)) {
                $message = 'Client cannot be deleted because there are active ' . implode(', ', $messageParts) . '.';
                return redirect()->route('client-index')->with('error', $message);
            }

            DB::beginTransaction();
            try {

                if ($client->profile_img) {
                    Storage::delete($client->profile_img);
                }
                if ($client->clientAddress) {
                    $client->clientAddress()->delete();
                }
                DB::table('branch_user')->where('user_id', $client->user_id)->delete();

                if ($client->user) {
                    centralizedUsersDelete($client->user->email);
                    $client->user->delete();
                }
                $client->delete();
                DB::commit();
                return redirect()->route('client-index')->with('success', 'Client deleted successfully!');
            } catch (\Exception $e) {
                DB::rollBack();
                \Log::error('While delete Client:' . $e->getMessage());
                return back()->with('error', 'Failed to delete client!');
            }
        } else {
            return back()->with('error', 'Client not found!');
        }
    }

    public function getStates($id)
    {
        $states = StateModel::
            where('country_id', $id)
            ->get();
        return response()->json($states);
    }

    public function getCities($id)
    {
        $cities = CityModel::where('state_id', $id)
            ->get();
        return response()->json($cities);
    }

}