<?php

namespace App\Http\Controllers;

use App\Models\AssembledItemModel;
use App\Models\BOMModel;
use App\Models\BranchModel;
use App\Models\CentralizedUsers;
use App\Models\CompanyModel;
use App\Models\ExpensesModel;
use App\Models\InvoiceModel;
use App\Models\ProductHistoryModel;
use App\Models\RightsModel;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class AdminController extends Controller
{
    protected $branch_id;
    protected $company_id;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->branch_id = session('branch_id');
            $this->company_id = session('comp_id');
            return $next($request);
        });
    }

    public function index()
    {
        if (in_array(Auth::user()->user_flg, ['A', 'SU'])) {
            $admins = User::whereIn('user_flg', ['S', 'A'])->where('branch_id', session('branch_id'))->orderBy('id', 'desc')->get();
        } else {
            $admins = [];
        }
        return view('admins.index', compact('admins'));
    }

    public function add()
    {
        $company_name = CompanyModel::where('id', $this->company_id)->pluck('name')->first();
        $branch_name = BranchModel::where('id', $this->branch_id)->pluck('name')->first();

        return view('admins.add', compact('company_name', 'branch_name'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            // 'email' => 'required|email|unique:users,email',
            'email' => [
                'required',
                'email',
                'unique:users,email',
                Rule::unique('second_db.users', 'email')
            ],
            'contact_no' => 'required|string|max:15|unique:users,phone',
            'password' => 'required|string|min:6|confirmed',
            'address' => 'required|string|max:255',
            'role' => 'required'
        ]);

        DB::beginTransaction();
        try {
            $profile_image_path = null;
            if ($request->hasFile('profile_image')) {
                $profile_image_path = storeFile($request->file('profile_image'), 'admin');
            }

            $user = User::create([
                'name' => $validated['first_name'],
                'last_name' => $validated['last_name'],
                'email' => $validated['email'],
                'phone' => $validated['contact_no'],
                'password' => Hash::make($validated['password']),
                'profile_img' => $profile_image_path,
                'location' => $validated['address'],
                'user_flg' => $validated['role'],
                'branch_id' => $this->branch_id,
                'active_status' => $request->input('active_status', 0),
            ]);

            if ($user) {
                DB::table('company_user')->insert([
                    'user_id' => $user->id,
                    'company_id' => $this->company_id,
                ]);

                DB::table('branch_user')->insert([
                    'user_id' => $user->id,
                    'branch_id' => $this->branch_id,
                ]);
            }

            centralizedUsers($validated['first_name'] . ' ' . $validated['last_name'], $validated['email'], $validated['contact_no'], );

            DB::commit();
            return redirect()->route('admins-index')->with('success', 'Admin added successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Something went wrong!')->withInput();
        }
    }

    public function edit($id)
    {
        $admin = User::findOrFail($id);
        return view('admins.edit', compact('admin'));
    }

    public function update(Request $request)
    {
        $admin = User::findOrFail($request->admin_id);
        $rules = [
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            // 'email' => 'required|email|unique:users,email,' . $request->admin_id,
            'email' => [
                'required',
                'email',
                'unique:users,email,' . $request->admin_id,
                Rule::unique('second_db.users', 'email')->ignore($admin->email, 'email')
            ],
            'contact_no' => 'required|string|unique:users,phone,' . $request->admin_id,
            'address' => 'required|string|max:255',
            'role' => 'required'
        ];

        $validated = $request->validate($rules);

        DB::beginTransaction();
        try {
            $admin = User::findOrFail($request->admin_id);
            $admin->name = $validated['first_name'];
            $admin->last_name = $validated['last_name'];
            $admin->email = $validated['email'];
            $admin->phone = $validated['contact_no'];
            $admin->location = $validated['address'];
            $admin->user_flg = $validated['role'];
            $admin->active_status = $request->input('active_status', 0);

            if ($request->hasFile('profile_image')) {
                $imagePath = storeFile($request->file('profile_image'), 'admin');
                $admin->profile_img = $imagePath;
            }

            $admin->save();
            DB::commit();

            return redirect()->route('admins-index')->with('success', 'Admin updated successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Admin not found!')->withInput();
        }
    }

    // public function delete($id)
    // {
    //     $admin = User::find($id);

    //     if ($admin) {
    //         centralizedUsersDelete($admin->email);
    //         $admin->delete();
    //         return redirect()->route('admins-index')->with('success', 'Admin deleted successfully!');
    //     }

    //     return back()->with('error', 'Admin not found!');
    // }

    public function delete($id)
    {
        $admin = User::findOrFail($id);

        $dependencies = [
            'Invoices' => InvoiceModel::where('created_by', $id)->count(),
            'Assemblies' => AssembledItemModel::where('created_by', $id)->count(),
            'Bill Of Material' => BOMModel::where('created_by', $id)->count(),
            'Rights' => RightsModel::where('created_by', $id)->count(),
            'Product' => ProductHistoryModel::where('created_by', $id)->count(),
            'Expense' => ExpensesModel::where('created_by', $id)->count(),
        ];

        $blockers = array_filter($dependencies);
        if (!empty($blockers)) {
            $message = 'User is referenced in: ' .
                implode(', ', array_keys($blockers));
            return back()->with('error', $message);
        }
        centralizedUsersDelete($admin->email);
        $admin->delete();
        return redirect()->route('admins-index')->with('success', 'Admin deleted successfully!');
    }
}